/*
    idea by Lukasz Wojtow <lw@wszia.edu.pl>
    code by Rafal Wijata http://www.wijata.com
*/

#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_log.h"
#include "http_main.h"
#include "http_protocol.h"
#include "util_script.h"
#include <stdio.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>

/* minimum of *id we can change to */
#define DIFFPRIVS_MIN_UID	100
#define DIFFPRIVS_MIN_GID	100

module MODULE_VAR_EXPORT diffprivs_module;

typedef struct excfg {
	char	*username;
	char	*groupname;
	char	*chrootdir; 	/* not full work yet */
	int	userid;		/* if userid is 0 - don't do anything */
	int	groupid;	/* same as above */
} excfg;

/* alloc */
static void *mod_diffprivs_server_config(pool *p, server_rec *s) {
excfg *cfg = (excfg *) ap_palloc(p,sizeof(excfg));
	cfg->username = NULL;
	cfg->groupname = NULL;
	cfg->chrootdir = NULL;
	cfg->userid = 0;		/* means don't switch */
	cfg->groupid = 0;		/* '' */
	return (void *) cfg;
};

/* runtime */
static int mod_diffprivs_first(request_rec *r) {
excfg *cfg;

	/* read our config */
	cfg = ap_get_module_config(r->server->module_config, &diffprivs_module);

	if (cfg->chrootdir) {
	    if (chroot(cfg->chrootdir)) {
		ap_log_rerror(APLOG_MARK,APLOG_ERR,r,
			"euid %d cannot chroot(%s)\n",
			geteuid(),cfg->chrootdir);
		return HTTP_FORBIDDEN;		
	    };
	    /* TODO: make chdir() */
	    /* TODO: make dir rearangement */
	};

	if (cfg->groupid) {
	    if( setgid(cfg->groupid) ) {
		ap_log_rerror(APLOG_MARK,APLOG_ERR,r,
			"egid %d cannot setgid(%d) [%s]\n",
			getegid(), cfg->groupid, cfg->groupname);
		return HTTP_FORBIDDEN;
	    };
	};

	if (cfg->userid) {
	    if( setuid(cfg->userid) ) {
		ap_log_rerror(APLOG_MARK,APLOG_ERR,r,
			"euid %d cannot setuid(%d) [%s]\n",
			geteuid(), cfg->userid, cfg->username);
		return HTTP_FORBIDDEN;
	    };
	};

	/* pass to next module */
	return DECLINED;
};

/* config */
static const char *privs_cmd(cmd_parms *cmd, void *mconfig, char *user, char *group, char *dir) {
struct	passwd *pw;
struct	group *gr;
excfg	*cfg;
int	tmp;
char	*endptr;
struct	stat st;	/* not a pointer ! */
static char	error[128];	/* also */

	cfg = ap_get_module_config(cmd->server->module_config,
						&diffprivs_module);

	if (user) {
	    if (*user == '#') {
		/* we have #UID eg. #500, why user must exist in passwd ? */
		endptr = NULL;
		tmp = strtol(user + 1, &endptr, 10);
		if (endptr && *endptr) {	/* was an error */
		    snprintf(error, sizeof(error), "Invalid UID [%s]", user);
		    return error;
		};
	    } else {	/* a username was given */
		pw = getpwnam(user);
		if (! pw) {
		    snprintf(error, sizeof(error), "No such user [%s]", user);
		    return error;
		};
		tmp = pw->pw_uid;
	    };
	    if (tmp >= DIFFPRIVS_MIN_UID) {
		cfg->username = user;
		cfg->userid = tmp;
	    } else {
		snprintf(error, sizeof(error),
		    "Min allowed UID is %d but used %d for [%s]",
		    DIFFPRIVS_MIN_UID, tmp, user);
		return error;
	    };
	};

	if (group) {
	    if (*group == '#') {	/* same as for user */
		endptr = NULL;
		tmp = strtol(group + 1, &endptr, 10);
		if (endptr && *endptr) {	/* was an error */
		    snprintf(error,sizeof(error), "Invalid GID [%s]", group);
		    return error;
		};
	    } else {
		gr = getgrnam(group);
		if (! gr) {
		    snprintf(error, sizeof(error), "No such group [%s]", group);
		    return error;
		};
		tmp = gr->gr_gid;
	    };
	    if (tmp >= DIFFPRIVS_MIN_GID) {
		cfg->groupname = group;
		cfg->groupid = tmp;
	    } else {
		snprintf(error, sizeof(error),
		    "Min allowed GID is %d but used %d for [%s]",
		    DIFFPRIVS_MIN_GID, tmp, group);
		return error;
	    };
	};

	if (dir) {	/* TODO: readlink() if main config says so */
	    if (! stat(dir, &st)) {
	    /* check whether dir exist,
	       of course user may delete it later,
	       but we'll not assume he create it for sure */
	        if (S_ISDIR(st.st_mode)) {
		    cfg->chrootdir = dir;
	        } else {
	    	    snprintf(error, sizeof(error), "Is not dir [%s]", dir);
	    	    return error;
	        };
	    } else {
	        snprintf(error, sizeof(error), "No such dir [%s]", dir);
	        return error;
	    };
	};

	return NULL;
};

static const command_rec mod_diffprivs_commands[] = {
	{
		"Privs", privs_cmd, NULL, RSRC_CONF, TAKE123,
		"Diffrent process privileges: uid [gid [chroot_dir]]"
	},
	{NULL}
};

static void mod_diffprivs_init(server_rec *r, pool *p) {
	excfg *cfg = ap_get_module_config(r->module_config, &diffprivs_module);
};

module MODULE_VAR_EXPORT diffprivs_module = {
	STANDARD_MODULE_STUFF,
	mod_diffprivs_init,
	NULL,
	NULL,
	mod_diffprivs_server_config,
	NULL,
	mod_diffprivs_commands,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	mod_diffprivs_first
};
